//------------------------------------------------------------
// Copyright Sandlot Games, 2007
// author: Michael Felice
// file: client_questHud.cs
// brief:
//    This file contains quest information as seen in the HUD.
// Quests appear on the screen in a row and can be highlighted
// or selected to view the task details of the quest.  The
// task details of a quest include a description and an image
// indication of the task has been completed or not.  Each
// quest name is also needed for each quest, which can be
// seen by selecting or highlighting a quest.
//------------------------------------------------------------

// this is the speed that quest icons slide to the left when
// a quest has been removed (or completed)
$QuestSlideSpeed = 40;

// the most tasks that will be listed in the quest hud
$QuestTaskMaxCount = 5;
$QuestTaskWaitTime = 3;
$QuestTaskFadeTime = 2;

// highlight colors refer to the icon's modulated selection color
// when a quest icon is selected or not selected
$QuestSelectHighlight = "1 1 1 1";
$QuestDeselectHighlight = "1 1 1 1";


//*************************************************************
//  HUD QUESTS INITIALIZATION, DESTRUCTION, SAVING, LOADING
//*************************************************************

// intitialize the quest hud properties
$QuestButtonSelected = 0;
$QuestWindowHud = 0;
$QuestTextName = 0;
$QuestWindowCount = 0;
$QuestWindowButtons = 0;
$QuestTaskButtons = 0;
$QuestSlideTimer = 0;
$QuestLastTime = 0;
$QuestBaseButton = 0;
$QuestNextButton = 0;
$QuestToggle = 0;

// this function will set up the quest window structures needed
// to display quest and task information about the quests (if
// this function is called multiple times, the quests will only
// be initialized once, and this function does not need to be
// called because quests are automatically intiailized when a
// new quest is added)
function InitHudQuests()
{
   // check if the quest window has been created, and create it
   // if it does not exist (along with the other quest components)
   if (isObject($QuestWindowHud) == false)
   {
      // initialize the quest window
      $QuestWindowHud = nameToId(QuestInfoGroup);
      $QuestWindowHud.visible = true;

      // update the highlight text
      $QuestTextName = nameToId(QuestTextName);
      $QuestTextName.stateUp = "";
      $QuestTextName.visible = true;
      
      // update objects that are used to display tasks and task completion
      for (%index = 0; %index < $QuestTaskMaxCount; %index++)
      {
         $QuestTaskButtons[%index, 0] = nameToId("QuestTaskText" @ %index);
         $QuestTaskButtons[%index, 1] = nameToId("QuestTaskImage" @ %index);
         $QuestTaskButtons[%index, 0].stateUp = "";
         $QuestTaskButtons[%index, 0].visible = false;
         $QuestTaskButtons[%index, 1].visible = false;
      }
      
      // update the location of the quest buttons
      $QuestBaseButton = nameToId(BaseQuestButton);
      $QuestNextButton = nameToId(NextQuestButton);
      $QuestBaseButton.visible = false;
      $QuestNextButton.visible = false;
      
      QuestToggle.visible = false;
      QuestToggle.input = false;
      $QuestToggle = new SLImage()
      {
         visible = "1";
         input = "1";
         onTop = "1";
         horizSizing = "relative";
         vertSizing = "relative";
         stateUp = QuestToggle.stateDown;
         stateFocus = QuestToggle.stateDown;
         stateDown = QuestToggle.stateDown;
         position = QuestToggle.position;
         extent = QuestToggle.extent;
         command = "TurnQuestsOff();";
      };
      TopbarHud.addGuiControl($QuestToggle);
   }
   
   // we also need a quest slide timer to ensure that the quest
   // icons are always in the correct position (or are sliding
   // to the correct position when they are not in the right position)
   if (isObject($QuestSlideTimer) == false)
   {
      $QuestSlideTimer = new SLEndlessTimer()
      {
         time = 0;
      };
      $QuestSlideTimer.notifyOnUpdate(SlideQuests);
   }
}

// this function will destroy any existing hud messages
function DestroyHudQuests()
{
   // delete the quest sliding timer
   if (isObject($QuestSlideTimer) == true)
   {
      $QuestSlideTimer.delete();
   }
   
   // hide any task details and images that were set
   for (%index = 0; %index < $QuestTaskMaxCount; %index++)
   {
      $QuestTaskButtons[%index, 0].stateUp = "";
      $QuestTaskButtons[%index, 1].visible = false;
   }
   
   // delete any quest buttons that were created
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      %button.delete();
   }
   
   // hide the quest name object
   $QuestTextName.stateUp = "";
   
   // reintitialize the quest hud properties
   $QuestButtonSelected = 0;
   $QuestWindowHud = 0;
   $QuestTextName = 0;
   $QuestWindowCount = 0;
   $QuestWindowButtons[0] = 0;
   $QuestTaskButtons[0] = 0;
   $QuestSlideTimer = 0;
   $QuestLastTime = 0;
   
   if (isObject($QuestToggle) == true)
   {
      $QuestToggle.delete();
   }
   $QuestToggle = 0;
}



//*************************************************************
//                       HUD QUEST SENDING
//*************************************************************

function GetButtonPosition(%index)
{
   %posX = getWord($QuestBaseButton.position, 0);
   %posY = getWord($QuestBaseButton.position, 1);
   %diffX = getWord($QuestNextButton.position, 0) - %posX;
   %diffY = getWord($QuestNextButton.position, 1) - %posY;
   
   return %posX + %index * %diffX @ " " @ %posY + %index * %diffY;
}

// this function is used to add a new quest icon that has the
// quest name label %questName, once this quest is added,
// tasks can be added for this quest
function AddQuest(%questName)
{
   // ensure that the hud quests are intiialized
   InitHudQuests();
   
   // if a quest already exists with the name, do not add it
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      if (%button.quest $= %questName)
      {
         return;
      }
   }
   
   // a new quest icon should appear and be added to the list of quests
   %button = new SLImage()
   {
      visible = "1";
      input = "1";
      onTop = "1";
      horizSizing = "relative";
      vertSizing = "relative";
      stateUp = $QuestBaseButton.stateUp;
      stateFocus = $QuestBaseButton.stateFocus;
      stateDown = $QuestBaseButton.stateFocus;
      position = GetButtonPosition($QuestWindowCount);
      extent = $QuestBaseButton.extent;
   };
   %button.FadeIn(1);
   %button.command = "SelectButton(" @ %button @ ");";
   %button.quest = %questName;
   %button.taskCount = 0;
   
   // handle quest selection (if the quest is added and should be
   // selected, then select it, otherwise, set it in the unselected
   // state
   if ($QuestButtonSelected == $QuestWindowCount)
   {
      %button.color = $QuestSelectHighlight;
      $QuestTextName.stateUp = %button.quest;
      
      %button.stateUp = $QuestBaseButton.stateDown;
      %button.stateFocus = $QuestBaseButton.stateDown;
      %button.stateDown = $QuestBaseButton.stateDown;
   }
   else
   {
      %button.color = $QuestDeselectHighlight;
   }

   // add the quest icon gui control and make sure the window
   // accuractely reports the new quest that was added
   $QuestWindowHud.addGuiControl(%button);
   $QuestWindowButtons[$QuestWindowCount] = %button;
   $QuestWindowCount++;
}

// this function should be called when a quest is complete.  it
// sends a message to the hud messages and removes the completed
// quest
function CompleteQuest(%questName)
{
   %msg = slgGetUIString("id_quest_complete");
   %msg = slgFormatUIString(%msg, %questName);
   alertPushMsg(%msg);
   RemoveQuest(%questName);
   playSFXSound("audio_questcomplete");
}

// this function is used to remove the quest that has been added
// with the name %questName.  This will also remove any tasks
// associated with the quest
function RemoveQuest(%questName)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      // we found the quest to remove the task to
      if (%button.quest $= %questName)
      {
         // deselect the current selection (if the quest to be removed
         // is currently selected)
         %select = $QuestButtonSelected;
         if ($QuestButtonSelected == %index)
         {
            SelectButton($QuestWindowButtons[%select]);
         }
         
         // remove the quest and slide any quests after this quest
         // down in the window button list
         %button.delete();
         for (%questID = %index + 1; %questID < $QuestWindowCount; %questID++)
         {
            $QuestWindowButtons[%questID - 1] = $QuestWindowButtons[%questID];
         }
         $QuestWindowCount--;
         
         // handle special quest selection handling to make
         // sure that if a new quest need to be selected, it is
         if (%select == %index)
         {
            if (%select == $QuestWindowCount && %select == 0)
            {
               $QuestButtonSelected = 0;               
            }
            else
            {
               if (%select == $QuestWindowCount) %select--;
               SelectButton($QuestWindowButtons[%select]);
            }
         }
         else if ($QuestButtonSelected > %index)
         {
            $QuestButtonSelected--;
         }

         break;
      }
   }
}

// replace the task information of the quest that already exists
function ReplaceQuestTaskBrief(%questName, %oldtask, %newtask)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      // we found the quest to add the task to
      if (%button.quest $= %questName)
      {
         for (%taskID = 0; %taskID < %button.taskCount; %taskID++)
         {
            // if the task name already exists, it might not be added
            %taskName = %button.tasks[%taskID, 0].stateUp;
            if (%taskName $= %oldtask)
            {
               // set the task
               %button.tasks[%taskID, 0].stateUp = %newtask;
               return;
            }
         }
      }
   }
}

// this function adds a task to a particular quest that has already
// been addded, where %questName was the quest added, and %task is
// the new task text to add to the quest
function AddQuestTask(%questName, %task)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      // we found the quest to add the task to
      if (%button.quest $= %questName)
      {
         for (%taskID = 0; %taskID < %button.taskCount; %taskID++)
         {
            // if the task name already exists, it might not be added
            %taskName = %button.tasks[%taskID, 0].stateUp;
            if (%taskName $= %task)
            {
               // if the task already exists and is unchecked, do
               // not add it to the quest log-- otherwise, it
               // is being removed, so allow adding a new one
               if (%button.tasks[%taskID, 1].selected == false)
               {
                  return;
               }
            }
         }
         
         // update the task count after getting the new task index
         %taskID = %button.taskCount;
         %button.taskCount++;
         
         %mimicButton = nameToId("QuestTaskImage" @ %taskID);
         %button.tasks[%taskID, 1] = new SLImage()
         {
            visible = "0";
            input = "0";
            stateUp = %mimicButton.stateUp;
            stateDown = %mimicButton.stateDown;
            color = %mimicButton.color;
            profile = %mimicButton.profile;
            horizSizing = %mimicButton.horizSizing;
            vertSizing = %mimicButton.vertSizing;
            position = %mimicButton.position;
            extent = %mimicButton.extent;
            iconOffset = %mimicButton.iconOffset;
            iconAlpha = %mimicButton.iconAlpha;
            iconEnabled = %mimicButton.iconEnabled;
            iconDisabled = %minicButton.iconDisabled;
         };
         QuestInfoGroup.addGuiControl(%button.tasks[%taskID, 1]);
         
         // add the task and tell it that it is not complete
         %mimicButton = nameToId("QuestTaskText" @ %taskID);
         %button.tasks[%taskID, 0] = new SLText()
         {
            visible = "0";
            input = "0";
            profile = %mimicButton.profile;
            horizSizing = %mimicButton.horizSizing;
            vertSizing = %mimicButton.vertSizing;
            position = %mimicButton.position;
            extent = %mimicButton.extent;
            iconOffset = %mimicButton.iconOffset;
            iconAlpha = %mimicButton.iconAlpha;
            iconEnabled = %mimicButton.iconEnabled;
            iconDisabled = %minicButton.iconDisabled;
            stateUp = %task;
         };
         QuestInfoGroup.addGuiControl(%button.tasks[%taskID, 0]);
         
         // if this particular task is visible, update the tasks
         if ($QuestButtonSelected == %index)
         {
            ShowTasks(%button);
         }
         break;
      }
   }
}

// this function check's a quests's task box, indicating to the
// player that the task has been completed (if all tasks have
// been completed at this point, the quest is removed from
// the quest list)
function CheckQuestTask(%questName, %task)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      // we found the quest to check the task of
      if (%button.quest $= %questName)
      {
         %checked = false;
         for (%taskID = 0; %taskID < %button.taskCount; %taskID++)
         {
            // if the task name already exists and has not been checked
            %taskName = %button.tasks[%taskID, 0].stateUp;
            if (%taskName $= %task && %button.tasks[%taskID, 1].selected == false)
            {
               // check this task
               %button.tasks[%taskID, 1].selected = true;
               %checked = true;
               
               // create the timer that will determine when the check
               // is removed from the list of tasks
               %timer = new SLTimer()
               {
                  time = $QuestTaskWaitTime;
               };
               %timer.notifyOnFire(FadeTaskTimer, %timer);
               %timer.button = %button;
               %timer.task = %taskName;
               %button.tasks[%taskID, 0].timer = %timer;
               %button.tasks[%taskID, 0].timerPhase = 1;

               // if this particular task is visible, update the tasks
               if ($QuestButtonSelected == %index)
               {
                  ShowTasks(%button);
               }
               else
               {
                  %button.Blink(2, 4, $QuestDeselectHighlight, $QuestSelectHighlight);
               }
               break;
            }
         }
         
         if (%checked == false)
         {
            return false;
         }
         
         return true;
      }
   }
   
   return false;
}

// this function removes a quest's task based on the quest name
// %questName and the %task, which is the task to remove
function RemoveQuestTask(%questName, %task)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      // we found the quest to add the task to
      if (%button.quest $= %questName)
      {
         for (%taskID = 0; %taskID < %button.taskCount; %taskID++)
         {
            // if the task name already exists, remove the task
            %taskName = %button.tasks[%taskID, 0].stateUp;
            if (%taskName $= %task)
            {
               %button.tasks[%taskId, 0].StopFade();
               %button.tasks[%taskId, 0].StopBlink();
               %button.tasks[%taskId, 0].delete();
               
               %button.tasks[%taskId, 1].StopFade();
               %button.tasks[%taskId, 1].StopBlink();
               %button.tasks[%taskId, 1].delete();
               
               // update all of the following tasks by sliding them down
               for (%taskNext = %taskID + 1; %taskNext < %button.taskCount; %taskNext++)
               {
                  %button.tasks[%taskNext - 1, 0] = %button.tasks[%taskNext, 0];
                  %button.tasks[%taskNext - 1, 1] = %button.tasks[%taskNext, 1];
               }
               %button.taskCount--;
               
               // if this particular task is visible, update the tasks
               if ($QuestButtonSelected == %index)
               {
                  ShowTasks(%button);
               }
               
               // check if there are any more tasks that need to be completed
               if (%button.taskCount == 0)
               {
                  CompleteQuest(%questName);
               }
               return;
            }
         }
         break;
      }
   }
}

function SLTimer::FadeTaskTimer(%timer)
{
   %button = %timer.button; 
   %task = %timer.task;
   
   // find the task id based on the known task
   %taskID = 0;
   for (%index = 0; %index < %button.taskCount; %index++)
   {
      %taskName = %button.tasks[%index, 0].stateUp;
      if (%taskName $= %task)
      {
         %taskID = %index;
         break;
      }
   }
   
   %button.tasks[%taskID, 0].FadeOut($QuestTaskFadeTime);
   %button.tasks[%taskID, 1].FadeOut($QuestTaskFadeTime);

   %newTimer = new SLTimer()
   {
      time = $QuestTaskFadeTime;
   };
   %newTimer.notifyOnFire(RemoveTaskTimer, %newTimer);
   %newTimer.quest = %button.quest;
   %newTimer.task = %task;
   
   %button.tasks[%taskID, 0].timer = %newTimer;
   %button.tasks[%taskID, 0].timerPhase = 2;
}

function SLTimer::RemoveTaskTimer(%timer)
{
   %quest = %timer.quest;
   %task = %timer.task;
   RemoveQuestTask(%quest, %task);
}

function clientCmdSelectQuestButton(%quest)
{
   SelectQuestButton(%quest);
}

function SelectQuestButton(%quest)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      if ($QuestWindowButtons[%index].quest $= %quest)
      {
         %oldButton = 0;
         %button = $QuestWindowButtons[%index];
         if ($QuestButtonSelected != -1)
         {
            %oldButton = $QuestWindowButtons[$QuestButtonSelected];
         }
         
         if ($QuestButtonSelected == %index) %index = -1;
         
         if (isObject(%oldButton) == true)
         {
            %oldButton.Blink(0.25, 0.5, $QuestSelectHighlight, $QuestDeselectHighlight);
            
            %oldButton.stateUp = $QuestBaseButton.stateUp;
            %oldButton.stateFocus = $QuestBaseButton.stateFocus;
            %oldButton.stateDown = $QuestBaseButton.stateFocus;
         }
         if (%index != -1)
         {
           %button.Blink(0.25, 0.5, $QuestDeselectHighlight, $QuestSelectHighlight);
           $QuestTextName.stateUp = %button.quest;
         }
         else
         {
            $QuestTextName.stateUp = "";
         }
         
         $QuestButtonSelected = %index;
         break;
      }
   }

   if ($QuestButtonSelected != -1)
   {
      %button.stateUp = $QuestBaseButton.stateDown;
      %button.stateFocus = $QuestBaseButton.stateDown;
      %button.stateDown = $QuestBaseButton.stateDown;
      
      CloseTasks(%oldButton);
      ShowTasks(%button);
   }
   else
   {
      %button.stateUp = $QuestBaseButton.stateUp;
      %button.stateFocus = $QuestBaseButton.stateFocus;
      %button.stateDown = $QuestBaseButton.stateFocus;
      
      CloseTasks(%button);
   }
}

// this function is called for any quest icon that is selected
function SelectButton(%button)
{
   if (isObject(%button) == false)
   {
      return;
   }
   
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      if ($QuestWindowButtons[%index] == %button)
      {
         %oldButton = 0;
         if ($QuestButtonSelected != -1)
         {
            %oldButton = $QuestWindowButtons[$QuestButtonSelected];
         }
         
         if ($QuestButtonSelected == %index) %index = -1;
         
         if (isObject(%oldButton) == true)
         {
            %oldButton.Blink(0.25, 0.5, $QuestSelectHighlight, $QuestDeselectHighlight);
            
            %oldButton.stateUp = $QuestBaseButton.stateUp;
            %oldButton.stateFocus = $QuestBaseButton.stateFocus;
            %oldButton.stateDown = $QuestBaseButton.stateFocus;
         }
         if (%index != -1)
         {
           %button.Blink(0.25, 0.5, $QuestDeselectHighlight, $QuestSelectHighlight);
           $QuestTextName.stateUp = %button.quest;
         }
         else
         {
            $QuestTextName.stateUp = "";
         }
         
         $QuestButtonSelected = %index;
         break;
      }
   }

   if ($QuestButtonSelected != -1)
   {
      %button.stateUp = $QuestBaseButton.stateDown;
      %button.stateFocus = $QuestBaseButton.stateDown;
      %button.stateDown = $QuestBaseButton.stateDown;
      
      CloseTasks(%oldButton);
      ShowTasks(%button);
   }
   else
   {
      %button.stateUp = $QuestBaseButton.stateUp;
      %button.stateFocus = $QuestBaseButton.stateFocus;
      %button.stateDown = $QuestBaseButton.stateFocus;
      
      CloseTasks(%button);
   }
}

// when a quest icon gains focus, make sure the highlight text
// displays that quest's name
function SLImage::buttonFocus(%this)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      if (%button == %this)
      {
         $QuestTextName.stateUp = %button.quest;
         return;
      }
   }
}

// if a quest loses focus, clear out the text highlight and
// replace the text highlight with the selected quest (if a
// quest has been selected)
function SLImage::buttonUnfocus(%this)
{
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      if (%button == %this)
      {
         $QuestTextName.stateUp = "";
         break;
      }
   }
   
   if ($QuestButtonSelected == -1)
   {
      CloseTasks(%this);
      return;
   }
   
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      if ($QuestButtonSelected == %index)
      {
         %button = $QuestWindowButtons[%index];
         $QuestTextName.stateUp = %button.quest;
         ShowTasks(%button);
         break;
      }
   }
}

// this function hides any tasks a quest has
function CloseTasks(%button)
{
   for (%index = 0; %index < %button.taskCount; %index++)
   {
      %button.tasks[%index, 0].visible = false;
      %button.tasks[%index, 1].visible = false;
   }
}

// this function displays any tasks a quest has
function ShowTasks(%button)
{
   CloseTasks(%button);
   
   for (%index = 0; %index < %button.taskCount; %index++)
   {
      %button.tasks[%index, 0].visible = true;
      %button.tasks[%index, 1].visible = true;
   }
}

// this function is called by the timer to ensure that each quest icon
// is in the proper location (sliding quest icons that are out of place)
function SlideQuests(%time)
{
   %interval = %time - $QuestLastTime;

   // slide all of the quest buttons to the right that needs to slide
   %offset = $QuestSlideSpeed * %interval;
   for (%index = 0; %index < $QuestWindowCount; %index++)
   {
      %button = $QuestWindowButtons[%index];
      
      %xPosGoal = getWord(GetButtonPosition(%index), 0);
      %xPos = getWord(%button.position, 0);
      if (%xPos > %xPosGoal) %xPos -= %offset;
      if (%xPos < %xPosGoal) %xPos = %xPosGoal;
      
      %yPos = getWord(%button.position, 1);
      %button.position = %xPos @ " " @ %yPos;
      
      // slide all of the quests tasks up that are currently not in place
      for (%taskIndex = 0; %taskIndex < %button.taskCount; %taskIndex++)
      {
         %textButton = %button.tasks[%taskIndex, 0];
         %textGoal = nameToId("QuestTaskText" @ %taskIndex);
         %textYPosGoal = getWord(%textGoal.position, 1);
         %textYPos = getWord(%textButton.position, 1);
         
         if (%textYPos > %textYPosGoal) %textYPos -= %offset;
         if (%textYPos < %textYPosGoal) %textYPos = %textYPosGoal;
         %textButton.position = getWord(%textButton.position, 0) @ " " @ %textYPos;

         %imageButton = %button.tasks[%taskIndex, 1];
         %imageGoal = nameToId("QuestTaskImage" @ %taskIndex);
         %imageYPosGoal = getWord(%imageGoal.position, 1);
         %imageYPos = getWord(%imageButton.position, 1);

         if (%imageYPos > %imageYPosGoal) %imageYPos -= %offset;
         if (%imageYPos < %imageYPosGoal) %imageYPos = %imageYPosGoal;
         %imageButton.position = getWord(%imageButton.position, 0) @ " " @ %imageYPos;
      }
   }

   $QuestLastTime = %time;
}

function shortcutToggleQuests()
{
   if (isObject($QuestToggle) == false)
   {
      return;
   }
   
   // get the function that should be called to toggle the quests
   %callList = strreplace($QuestToggle.command, "(", " ");
   %function = getWord(%callList, 0);
   call(%function);
}

function TurnQuestsOff()
{
   if (isObject($QuestToggle) == false)
   {
      return;
   }
   
   QuestInfoGroup.visible = false;
   
   $QuestToggle.stateUp = QuestToggle.stateUp;
   $QuestToggle.stateFocus = QuestToggle.stateFocus;
   $QuestToggle.stateDown = QuestToggle.stateFocus;
   $QuestToggle.command = "TurnQuestsOn();";
}

function TurnQuestsOn()
{
   if (isObject($QuestToggle) == false)
   {
      return;
   }
   
   QuestInfoGroup.visible = true;
   
   $QuestToggle.stateUp = QuestToggle.stateDown;
   $QuestToggle.stateFocus = QuestToggle.stateDown;
   $QuestToggle.stateDown = QuestToggle.stateDown;
   $QuestToggle.command = "TurnQuestsOff();";
}
